<?php 
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class SqlConsoleController extends Controller
{ 
    public function index()
    {
        // Récupération des tables pour le <select>
        $tables = array_map('current', DB::select('SHOW TABLES'));
        return view('sql-console', [
            'tables' => $tables
        ]);
    }

    public function execute(Request $request)
    {
        $sql = trim($request->input('sql'));
        $allowDanger = $request->has('allowDanger');
        $page = $request->input('page', 1);

        // Sauvegarde du dernier SQL saisi, même si erreur
        session(['last_query' => $sql]);

        // Vérifie les requêtes dangereuses
        if (!$allowDanger && preg_match('/\b(drop|truncate|delete|alter)\b/i', $sql)) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['danger' => '❌ Requête potentiellement dangereuse bloquée.']);
        }

        try {
            $results = [];

            if (preg_match('/^select/i', $sql)) {
                // Gestion pagination
                $perPage = 20;
                $offset = ($page - 1) * $perPage;

                $paginated = DB::select(DB::raw("$sql LIMIT $perPage OFFSET $offset"));
                $results = json_decode(json_encode($paginated), true);

                // Calcul total
                $countQuery = "SELECT COUNT(*) as total FROM ($sql) AS sub";
                $total = DB::select($countQuery)[0]->total ?? count($results);

                session([
                    'paginate' => [
                        'total_pages' => ceil($total / $perPage),
                        'current' => $page
                    ],
                    'results' => $results
                ]);
            } else {
                // Requêtes d’action (INSERT, UPDATE, etc.)
                DB::statement($sql);
                session([
                    'results' => [['message' => '✅ Requête exécutée avec succès.']]
                ]);
                session()->forget('paginate');
            }

            // Gestion de l’historique
            $history = session('history', []);
            array_unshift($history, $sql);
            session(['history' => array_slice($history, 0, 10)]);

            return redirect()->back()->withInput();

        } catch (\Exception $e) {
            // En cas d’erreur SQL
            return redirect()
                ->back()
                ->withInput() // garde le contenu du champ
                ->withErrors(['sql' => '⚠️ Erreur SQL : ' . $e->getMessage()]);
        }
    }

    public function exportCsv()
    {
        $results = session('results', []);
        if (empty($results)) return redirect()->back();

        $headers = array_keys($results[0]);
        $filename = "export_" . date("Ymd_His") . ".csv";

        header("Content-Type: text/csv");
        header("Content-Disposition: attachment; filename=$filename");

        $handle = fopen('php://output', 'w');
        fputcsv($handle, $headers);
        foreach ($results as $row) {
            fputcsv($handle, $row);
        }
        fclose($handle);
        exit;
    }

    public function showStructure($table)
    {
        try {
            $info = DB::select("DESCRIBE `$table`");
            $output = "Structure de `$table`:\n";
            foreach ($info as $column) {
                $output .= "- {$column->Field} ({$column->Type})\n";
            }
            return response($output);
        } catch (\Exception $e) {
            return response("Erreur : " . $e->getMessage(), 500);
        }
    }

    public function showCreate($table)
    {
        try {
            $stmt = DB::select("SHOW CREATE TABLE `$table`");
            return response($stmt[0]->{'Create Table'} ?? $stmt[0]->{'Create View'});
        } catch (\Exception $e) {
            return response("Erreur : " . $e->getMessage(), 500);
        }
    }
}
