<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class CarteController extends Controller
{
    protected function index()
    {
        return view('profile_eleve.paiement_carte');
    }

    protected function checkAuth()
    {
        if (!session('logged_in')) {
            // abort(401, 'Unauthorized');
        }
    }
    
    public function elevesTable(Request $req)
    {
        $this->checkAuth();
        $ecole_id = \App\Config::getEcoleId();
		$id_annee = \App\Config::getIdAnnee();
    
        $filter = $req->query('filter', null);
        $classe_id = $req->query('classe', null);
    
        $query = DB::table('bulletin_ms_eleves as el')
            ->join('enfants as e', 'el.enfant_id', '=', 'e.id')
            ->join('bulletin_ms_classes as cl', 'cl.id', '=', 'el.ms_classe_id')
            ->where('el.ecole_id', $ecole_id)
            ->where('el.ms_annee_scolaire_id', $id_annee)
            ->select(
                'el.id',
                DB::raw("CONCAT(e.nom,' ',e.postnom,' ',e.prenom) as nom"),
                'cl.classe',
                'el.montant_carte',
                'el.payer_carte_le',
                'el.carte_imprimee_le',
                'el.carte_remise_le'
            );
    
        if ($classe_id) {
            $query->where('el.ms_classe_id', $classe_id);
        }
    
        switch ($filter) {
            case 'non_ordre':
                $query->whereNull('el.montant_carte');
                break;
            case 'en_ordre':
                $query->whereNotNull('el.montant_carte');
                break;
            case 'imprimee':
                $query->whereNotNull('el.carte_imprimee_le');
                break;
            case 'remise':
                $query->whereNotNull('el.carte_remise_le');
                break;
            case 'paye_pas_carte':
                $query->whereNotNull('el.montant_carte')->whereNull('el.carte_imprimee_le');
                break;
        }
    
        $data = $query->orderBy('cl.classe')->orderBy('nom')->get();
    
        // ajouter numéro par classe
        $grouped = $data->groupBy('classe');
        $result = [];
        foreach ($grouped as $classe => $liste) {
            $num = 1;
            foreach ($liste as $e) {
                $e->numero = $num++;
                $result[] = $e;
            }
        }
    
        return response()->json($result);
    }

    public function elevesSelect2(Request $req)
    {
        $this->checkAuth();
        $ecole_id = \App\Config::getEcoleId();
		$id_annee = \App\Config::getIdAnnee();
    
        $search = $req->query('q', null); // texte recherché
        $classe_id = $req->query('classe', null);
    
        $query = DB::table('bulletin_ms_eleves as el')
            ->join('enfants as e', 'el.enfant_id', '=', 'e.id')
            ->join('bulletin_ms_classes as cl', 'cl.id', '=', 'el.ms_classe_id')
            ->where('el.ecole_id', $ecole_id)->where("ms_annee_scolaire_id",$id_annee)
            ->select(
                'el.id',
                DB::raw("CONCAT(e.nom,' ',e.postnom,' ',e.prenom) as nom"),
                'cl.classe'
            );
    
        if ($classe_id) {
            $query->where('el.ms_classe_id', $classe_id);
        }
    
        if ($search) {
            $search = str_replace(" ", "%", $search);
        
            $query->where(function($q) use ($search) {
                $q->where(DB::raw("CONCAT(e.nom, ' ', e.postnom, ' ', e.prenom)"), 'LIKE', "%{$search}%");
            });
        }

    
        $data = $query->orderBy('cl.classe')->orderBy('nom')->get();
    
        $results = $data->map(function($e) {
            return [
                'id' => $e->id,
                'text' => $e->nom . ' — ' . $e->classe
            ];
        });

        return response()->json(['results' => $results]);
    }

    /**
     * GET /api/gst-carte/eleves
     */
    public function eleves(Request $req)
    {
        $this->checkAuth();
        $ecole_id = \App\Config::getEcoleId();
		$id_annee = \App\Config::getIdAnnee();
    
        $filter = $req->query('filter', null);
        $classe_id = $req->query('classe', null);
        $search = $req->query('q', null); // paramètre recherche Select2
    
        $query = DB::table('bulletin_ms_eleves as el')
            ->join('enfants as e', 'el.enfant_id', '=', 'e.id')
            ->join('bulletin_ms_classes as cl', 'cl.id', '=', 'el.ms_classe_id')
            ->where('el.ecole_id', $ecole_id)->where("ms_annee_scolaire_id",$id_annee)
            ->select(
                'el.id',
                DB::raw("CONCAT(e.nom,' ',e.postnom,' ',e.prenom) as nom"),
                'cl.classe',
                'el.montant_carte',
                'el.payer_carte_le',
                'el.carte_imprimee_le',
                'el.carte_remise_le'
            );
    
        // Filtrer par classe si fourni
        if ($classe_id) {
            $query->where('el.ms_classe_id', $classe_id);
        }
    
        // Filtrer par texte si fourni (Select2)
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('e.nom', 'like', "%$search%")
                  ->orWhere('e.postnom', 'like', "%$search%")
                  ->orWhere('e.prenom', 'like', "%$search%");
            });
        }
    
        // Filtres spécifiques
        switch ($filter) {
            case 'non_ordre':
                $query->whereNull('el.montant_carte');
                break;
            case 'en_ordre':
                $query->whereNotNull('el.montant_carte');
                break;
            case 'imprimee':
                $query->whereNotNull('el.carte_imprimee_le');
                break;
            case 'remise':
                $query->whereNotNull('el.carte_remise_le');
                break;
            case 'paye_pas_carte':
                $query->whereNotNull('el.montant_carte')->whereNull('el.carte_imprimee_le');
                break;
        }

        // Trier par classe puis nom
        $data = $query->orderBy('cl.classe')->orderBy('nom')->get();
    
        // Ajouter numéro d'ordre par classe
        $grouped = $data->groupBy('classe');
        $result = [];
        foreach ($grouped as $classe => $liste) {
            $num = 1;
            foreach ($liste as $e) {
                $e->numero = $num++;
                $result[] = $e;
            }
        }
    
        // Stats
        $today = Carbon::today()->toDateString();
    
        $totalPaye = DB::table('bulletin_ms_eleves')
            ->where('ecole_id', $ecole_id)->where("ms_annee_scolaire_id",$id_annee)
            ->sum('montant_carte');
    
        $payeAuj = DB::table('bulletin_ms_eleves')
            ->where('ecole_id', $ecole_id)->where("ms_annee_scolaire_id",$id_annee)
            ->where('payer_carte_le', $today)
            ->sum('montant_carte');
    
        return response()->json([
            'eleves' => $result,
            'totalPaye' => (int)$totalPaye,
            'payeAuj' => (int)$payeAuj
        ]);
    }


    /**
     * POST /api/gst-carte/updateCarte
     */
    public function updateCarte(Request $req)
    {
        $this->checkAuth();
    
        $validator = Validator::make($req->all(), [
            'type' => 'required|in:payer,imprimer,remise',
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer',
            'date' => 'required|date|before_or_equal:today',
            'montant' => 'nullable|numeric|min:0',
            'force' => 'nullable' //|boolean case à cocher pour forcer la modification
        ]);
    
        if ($validator->fails()) {
            // On récupère les messages d'erreur de validation et on les transforme en tableau simple
            $errors = $validator->errors()->all();
        
            return response()->json([
                'status' => 'error',
                'messages' => $errors
            ], 422);
        }

        
    
        $type = $req->input('type');
        $ids = $req->input('ids');
        $date = $req->input('date') ? Carbon::parse($req->input('date'))->toDateString() : null;
        $montant = $req->input('montant');
        $force = $req->input('force', false);
        $force = $force=="false"?false:$force;
        $messages = []; // tableau des messages de retour
    
        try {
            DB::beginTransaction();
    
            foreach ($ids as $id) {
                // jointure avec la table enfants
                $eleve = DB::table('bulletin_ms_eleves as b')
                    ->join('enfants as e', 'b.enfant_id', '=', 'e.id')
                    ->select('b.*', 'e.nom', 'e.postnom', 'e.prenom')
                    ->where('b.id', $id)
                    ->first();
    
                if (!$eleve) {
                    $messages[] = "Élève ID $id introuvable.";
                    continue;
                }
    
                $fullName = trim("{$eleve->nom} {$eleve->postnom} {$eleve->prenom}");
    
                // Vérification selon le type
                if ($type === 'payer') {
                    if (!$force && $eleve->montant_carte) {
                        $messages[] = "Élève {$fullName} : paiement déjà enregistré le {$eleve->payer_carte_le}.";
                        continue;
                    }
    
                    DB::table('bulletin_ms_eleves')
                        ->where('id', $id)
                        ->update([
                            'payer_carte_le' => $date,
                            'montant_carte' => $montant // ?? $eleve->montant_carte
                        ]);
    
                    $messages[] = "Élève {$fullName} : paiement mis à jour.";
                } elseif ($type === 'imprimer') {
                    if (!$force && $eleve->carte_imprimee_le) {
                        $messages[] = "Élève {$fullName} : carte déjà imprimée le {$eleve->carte_imprimee_le}.";
                        continue;
                    }
    
                    DB::table('bulletin_ms_eleves')
                        ->where('id', $id)
                        ->update(['carte_imprimee_le' => $date]);
    
                    $messages[] = "Élève {$fullName} : impression enregistrée.";
                } elseif ($type === 'remise') {
                    if (!$force && $eleve->carte_remise_le) {
                        $messages[] = "Élève {$fullName} : carte déjà remise le {$eleve->carte_remise_le}.";
                        continue;
                    }
    
                    DB::table('bulletin_ms_eleves')
                        ->where('id', $id)
                        ->update(['carte_remise_le' => $date]);
    
                    $messages[] = "Élève {$fullName} : remise enregistrée.";
                }
            }
    
            DB::commit();
        } catch (\Exception $ex) {
            DB::rollBack();
            return response()->json([
                'error' => 'Erreur lors de la mise à jour : ' . $ex->getMessage()
            ], 500);
        }
    
        return response()->json([
            'status' => 'success',
            'messages' => $messages
        ]);
    }



    /**
     * GET /api/gst-carte/montantParDate
     */
    public function montantParDate(Request $req)
    {
        $this->checkAuth();

        $d = $req->query('d');

        if (!$d) return response()->json(['montant' => 0]);

        $date = Carbon::parse($d)->toDateString();

        $montant = DB::table('bulletin_ms_eleves')
            ->where('payer_carte_le', $date)
            ->sum('montant_carte');

        return response()->json(['montant' => (int)$montant]);
    }
    
    public function stats(Request $req)
    {
        $this->checkAuth();
    
        $ecole_id = \App\Config::getEcoleId();
		$id_annee = \App\Config::getIdAnnee();
    
        $total = DB::table('bulletin_ms_eleves')
            ->where('ecole_id', $ecole_id)->where("ms_annee_scolaire_id",$id_annee)
            ->whereNotNull('montant_carte')
            ->sum('montant_carte');
    
        $today = \Carbon\Carbon::today()->toDateString();
    
        $aujourdhui = DB::table('bulletin_ms_eleves')
            ->where('ecole_id', $ecole_id)
            ->where('payer_carte_le', $today)
            ->sum('montant_carte');
    
        return response()->json([
            'total' => (int)$total,
            'aujourdhui' => (int)$aujourdhui
        ]);
    }

}